#ifndef _RDS_IB_H
#define _RDS_IB_H 

#include <linux/mutex.h>

#include <rdma/ib_verbs.h>
#include <rdma/rdma_cm.h>
#include "rds.h"

#define RDS_IB_RESOLVE_TIMEOUT_MS	5000

/*
 * IB posts RDS_FRAG_SIZE fragments of pages to the receive queues to 
 * try and minimize the amount of memory tied up both the device and
 * socket receive queues.
 */
/* page offset of the final full frag that fits in the page */
#define RDS_PAGE_LAST_OFF (((PAGE_SIZE  / RDS_FRAG_SIZE) - 1) * RDS_FRAG_SIZE)
struct rds_page_frag {
	struct list_head	f_item;
	struct page		*f_page;
	unsigned long		f_offset;
};

struct rds_ib_incoming {
	struct list_head	ii_frags;
	struct rds_incoming	ii_inc;
};

struct rds_ib_connect_private {
	__be32	dp_saddr;
	__be32	dp_daddr;
};

struct rds_ib_send_work {
	struct rds_message	*s_rm;
	struct ib_send_wr	s_wr;
	struct ib_sge		s_sge[2];
};

struct rds_ib_recv_work {
	struct rds_ib_incoming 	*r_ibinc;
	struct rds_page_frag	*r_frag;
	struct ib_recv_wr	r_wr;
	struct ib_sge		r_sge[2];
};

struct rds_ib_work_ring {
	spinlock_t	w_lock;
	u32		w_nr;
	u32		w_next_free;
	u32		w_nr_free;
};

struct rds_ib_ack {
	__be64		a_seq;
} __attribute__((packed));

struct rds_ib_connection {
	/* alphabet soup, IBTA style */
	struct rdma_cm_id	*i_cm_id;
	struct ib_pd		*i_pd;
	struct ib_mr		*i_mr;
	struct ib_cq		*i_send_cq;
	struct ib_cq		*i_recv_cq;
	
	/* stops tx/rx posting */
	int 			i_wc_err;

	/* tx */
	struct rds_ib_work_ring	i_send_ring;
	struct rds_message	*i_rm;
	struct rds_header	*i_send_hdrs;
	dma_addr_t 		i_send_hdrs_dma;
	struct rds_ib_send_work *i_sends;

	/* rx */
	struct mutex		i_recv_mutex;
	struct rds_ib_work_ring	i_recv_ring;
	struct rds_ib_incoming	*i_ibinc;
	u32			i_recv_data_rem;
	struct rds_header	*i_recv_hdrs;
	dma_addr_t 		i_recv_hdrs_dma;
	struct rds_ib_recv_work *i_recvs;
	struct rds_page_frag	i_frag;
	dma_addr_t 		i_addr;

	/* sending acks */
	spinlock_t		i_ack_lock;
	int			i_ack_in_flight;
	u64			i_ack_next;
	struct rds_ib_ack	*i_ack;
	struct ib_send_wr	i_ack_wr;
	struct ib_sge		i_ack_sge;
	dma_addr_t 		i_ack_dma;
 
 	/* sending congestion bitmaps */
 	struct scatterlist	i_map_sg[RDS_CONG_MAP_PAGES];
 	unsigned int		i_map_count;
};

struct rds_ib_statistics {
	unsigned long	s_ib_connect_raced;
	unsigned long	s_ib_listen_closed_stale;
	unsigned long	s_ib_tx_cq_call;
	unsigned long	s_ib_tx_cq_event;
	unsigned long	s_ib_tx_ring_full;
	unsigned long	s_ib_tx_sg_mapping_failure;
	unsigned long	s_ib_rx_cq_call;
	unsigned long	s_ib_rx_cq_event;
	unsigned long	s_ib_rx_refill_from_cq;
	unsigned long	s_ib_rx_refill_from_thread;
	unsigned long	s_ib_ack_sent;
	unsigned long	s_ib_ack_send_failure;
	unsigned long	s_ib_ack_send_delayed;
	unsigned long	s_ib_ack_received;
};

extern struct workqueue_struct *rds_ib_wq;

/* ib.c */
extern struct rds_transport rds_ib_transport;

/* ib_cm.c */
int rds_ib_conn_alloc(struct rds_connection *conn, gfp_t gfp);
void rds_ib_conn_free(void *arg);
int rds_ib_conn_connect(struct rds_connection *conn);
void rds_ib_conn_shutdown(struct rds_connection *conn);
void rds_ib_state_change(struct sock *sk);
int __init rds_ib_listen_init(void);
void rds_ib_listen_stop(void);

/* ib_recv.c */
int __init rds_ib_recv_init(void);
void rds_ib_recv_exit(void);
int rds_ib_recv(struct rds_connection *conn);
void rds_ib_inc_purge(struct rds_incoming *inc);
void rds_ib_inc_free(struct rds_incoming *inc);
int rds_ib_inc_copy_to_user(struct rds_incoming *inc, struct iovec *iov,
			     size_t size);
void rds_ib_recv_cq_comp_handler(struct ib_cq *cq, void *context);
void rds_ib_recv_init_ring(struct rds_ib_connection *ic);
void rds_ib_recv_clear_ring(struct rds_ib_connection *ic);
void rds_ib_recv_init_ack(struct rds_ib_connection *ic);
void rds_ib_ack_send_complete(struct rds_ib_connection *ic);

/* ib_ring.c */
void rds_ib_ring_init(struct rds_ib_work_ring *ring, u32 nr);
u32 rds_ib_ring_alloc(struct rds_ib_work_ring *ring, u32 val, u32 *pos);
u32 rds_ib_ring_free(struct rds_ib_work_ring *ring, u32 val);
void rds_ib_ring_unalloc(struct rds_ib_work_ring *ring, u32 val);
int rds_ib_ring_empty(struct rds_ib_work_ring *ring);
u32 rds_ib_ring_oldest(struct rds_ib_work_ring *ring);
u32 rds_ib_ring_completed(struct rds_ib_work_ring *ring, u32 wr_id, u32 oldest);
extern wait_queue_head_t rds_ib_ring_empty_wait;

/* ib_send.c */
int rds_ib_xmit(struct rds_connection *conn, struct rds_message *rm,
	        unsigned int hdr_off, unsigned int sg, unsigned int off);
void rds_ib_send_cq_comp_handler(struct ib_cq *cq, void *context);
void rds_ib_send_init_ring(struct rds_ib_connection *ic);
void rds_ib_send_clear_ring(struct rds_ib_connection *ic);
int rds_ib_xmit_cong_map(struct rds_connection *conn,
			 struct rds_cong_map *map, unsigned long offset);

/* ib_stats.c */
RDS_DECLARE_PER_CPU(struct rds_ib_statistics, rds_ib_stats);
#define rds_ib_stats_inc(member) rds_stats_inc_which(rds_ib_stats, member)
unsigned int rds_ib_stats_info_copy(struct rds_info_iterator *iter,
				    unsigned int avail);

/* ib_sysctl.c */
int __init rds_ib_sysctl_init(void);
void rds_ib_sysctl_exit(void);
extern unsigned long rds_ib_sysctl_max_send_wr;
extern unsigned long rds_ib_sysctl_max_recv_wr;
extern unsigned long rds_ib_sysctl_max_unsig_wrs;
extern ctl_table rds_ib_sysctl_table[];

#endif
